var SpecialCalculator;

SpecialCalculator = (function() {

    /**
     * CSS class to be applied to available perks
     * @type {string}
     */
    SpecialCalculator.availablePerkClass = 'success';

    /**
     * URL of the calculator page
     * @type {string}
     */
    SpecialCalculator.baseUrl = 'http://www.nukahub.com/tools/special';

    /**
     * @constructor
     */
    function SpecialCalculator() {
        this.special = { s: 1, p: 1, e: 1, c: 1, i: 1, a: 1, l: 1 };
        this.pointsAllocated = 0;
    }

    /**
     * Gets the calculated Hit Points value
     * @returns {number}
     */
    SpecialCalculator.prototype.getHitPoints = function() {
        return 80 + (this.special.e * 5);
    };

    /**
     * Gets the calculated Action Points value
     * @returns {number}
     */
    SpecialCalculator.prototype.getActionPoints = function() {
        return 60 + (this.special.a * 10);
    };

    /**
     * Gets the calculated Carry Weight value
     * @returns {number}
     */
    SpecialCalculator.prototype.getCarryWeight = function() {
        return 200 + (this.special.s * 10);
    };

    /**
     * Gets the URL for the current build
     * @returns {string}
     */
    SpecialCalculator.prototype.getBuildUrl = function() {
        return SpecialCalculator.baseUrl
            + '?s=' + this.special.s
            + '&p=' + this.special.p
            + '&e=' + this.special.e
            + '&c=' + this.special.c
            + '&i=' + this.special.i
            + '&a=' + this.special.a
            + '&l=' + this.special.l;
    };

    /**
     * Loads the initial display and event handlers
     */
    SpecialCalculator.prototype.load = function() {
        var calc = this;

        calc.loadFromQueryString();
        calc.updateUrl();

        calc.updateSpecial();
        calc.updateRequirements();
        calc.updateDerivedStats();
        calc.updatePerks();

        $('.sub-special').on('click', function(event) {
            event.preventDefault();

            var specialType = $(this).data('special');
            var specialValue = calc.getSpecial(specialType);

            if (specialValue > 1) {
                calc.setSpecial(specialType, specialValue - 1);
                calc.setPointsAllocated(calc.pointsAllocated - 1);
                calc.updateUrl();
            }

            $(this).blur();
        });

        $('.add-special').on('click', function(event) {
            event.preventDefault();

            var specialType = $(this).data('special');
            var specialValue = calc.getSpecial(specialType);

            if (specialValue < 10) {
                calc.setSpecial(specialType, specialValue + 1);
                calc.setPointsAllocated(calc.pointsAllocated + 1);
                calc.updateUrl();
            }

            $(this).blur();
        });

        $('.reset-special').on('click', function(event) {
            event.preventDefault();
            calc.reset();
            calc.updateUrl();
            $(this).blur();
        });
    };

    /**
     * Loads initial special values from the query string
     */
    SpecialCalculator.prototype.loadFromQueryString = function() {
        var url = document.location.href;
        var queryStringStart = url.indexOf('?');

        if (queryStringStart > 0) {
            var queryString = url.substring(queryStringStart + 1);
            var queryStringParams = queryString.split('&');

            for (var i in queryStringParams) {
                var pair = queryStringParams[i].split('=');

                if (pair.length != 2) {
                    continue;
                }

                var type = pair[0];
                var value = parseInt(pair[1]);

                if (isNaN(value) || !(type in this.special)) {
                    continue;
                }

                if (value < 1) {
                    value = 1;
                }

                if (value > 10) {
                    value = 10;
                }

                this.special[type] = value;
                this.pointsAllocated += value - 1;
            }
        }
    };

    /**
     * Resets values to their defaults
     */
    SpecialCalculator.prototype.reset = function() {
        for (var specialType in this.special) {
            this.setSpecial(specialType, 1);
        }
        this.setPointsAllocated(0);
    };

    /**
     * Gets value of given special type
     * @param specialType
     * @returns {number}
     */
    SpecialCalculator.prototype.getSpecial = function(specialType) {
        return this.special[specialType];
    };

    /**
     * Sets value of given special type and updates display
     * @param specialType
     * @param specialValue
     */
    SpecialCalculator.prototype.setSpecial = function(specialType, specialValue) {
        this.special[specialType] = specialValue;
        this.updateSpecial(specialType);
        this.updateDerivedStats();
        this.updatePerks(specialType);
    };

    /**
     * Sets the amount of special points allocated and updates requirements display
     * @param amount
     */
    SpecialCalculator.prototype.setPointsAllocated = function(amount) {
        this.pointsAllocated = amount;
        this.updateRequirements();
    };

    /**
     * Updates display of given special type (or all if no type given)
     * @param specialType
     */
    SpecialCalculator.prototype.updateSpecial = function(specialType) {
        if (specialType === undefined) {
            for (specialType in this.special) {
                this.updateSpecial(specialType);
            }
        } else {
            $('.special-value[data-special="' + specialType + '"]').text(this.special[specialType]);
        }
    };

    /**
     * Updates display of remaining character creation points and minimum required level
     */
    SpecialCalculator.prototype.updateRequirements = function() {
        var pointsUnder21 = this.pointsAllocated <= 21;

        $('.creation-points-remaining').toggleClass('hidden', !pointsUnder21);
        $('.minimum-level').toggleClass('hidden', pointsUnder21);

        if (pointsUnder21) {
            $('.creation-points-remaining-value').text(21 - this.pointsAllocated);
        } else {
            $('.minimum-level-value').text(this.pointsAllocated - 20);
        }
    };

    /**
     * Updates display of derived stats
     */
    SpecialCalculator.prototype.updateDerivedStats = function() {
        $('.hp-value').text(this.getHitPoints());
        $('.ap-value').text(this.getActionPoints());
        $('.cw-value').text(this.getCarryWeight());
    };

    /**
     * Updates display of available perks for given special type (or all if no type given)
     * @param specialType
     */
    SpecialCalculator.prototype.updatePerks = function(specialType) {
        if (specialType === undefined) {
            for (specialType in this.special) {
                this.updatePerks(specialType);
            }
        } else {
            for (var i = 1; i <= 10; ++i) {
                $('.perk[data-special="' + specialType + '"][data-perk="' + i + '"]')
                    .toggleClass(SpecialCalculator.availablePerkClass, this.special[specialType] >= i);
            }
        }
    };

    /**
     * Updates the build link's href attribute and browser address (via HTML5 history API)
     */
    SpecialCalculator.prototype.updateUrl = function() {
        var url = this.getBuildUrl();

        $('a.build-link').attr('href', url);

        if (history.replaceState && document.location.href.indexOf(SpecialCalculator.baseUrl) == 0) {
            history.replaceState(null, null, url);
        }
    };

    return SpecialCalculator;
})();

var calc = new SpecialCalculator();
calc.load();